/*
 *   This is a starter TADS source file.  This is a complete TADS game
 *   that you can compile and run.
 *   
 *   To compile this game in TADS Workbench, open the "Build" menu and
 *   select "Compile for Debugging."  To run the game, after compiling it,
 *   open the "Debug" menu and select "Go."
 *   
 *   Please note that this file contains considerably more than the
 *   minimal set of definitions necessary to create a working game; this
 *   file has numerous examples meant to help you start making progress on
 *   your game more quickly.  If you want a truly minimal set of
 *   definitions, create a new game in TADS Workbench, and choose the
 *   "advanced" version when asked for the type of starter game to create.
 *   
 *   Copyright (c) 1999 by Michael J. Roberts.  Permission is granted to
 *   anyone to copy and use this file for any purpose.  
 */

/*
 *   Set a flag indicating that we want to use the HTML version of the
 *   status line code.  This must be set before we include adv.t, because
 *   adv.t needs to know what kind of status line we want. 
 */
#define USE_HTML_STATUS

/*
 *   Set another flag, this time indicating that we want to use the HTML
 *   version of the command prompt display code.  This code is in std.t,
 *   so we must define this flag before we include std.t.  
 */
#define USE_HTML_PROMPT

/*
 *   Include the TADS "adventure" library.  This library defines object
 *   classes that we can use as building blocks for creating the game
 *   objects.
 */
#include <adv.t>

/*
 *   Include the TADS "standard" library.  This contains default object
 *   definitions for several objects that are required in any TADS game.
 *   As you customize your game, you might eventually want to replace the
 *   definitions in std.t with your own definitions, but we can at least
 *   get started with the default versions.  
 */
#include <std.t>

/*
 *   Replace the function "commontInit" defined in std.t so that we turn
 *   on HTML mode.
 */
replace commonInit: function
{
    /* display the special code sequence to turn on HTML recognition */
    "\H+";
}


/*
 *   We must define the initial location, where the player starts the
 *   game.  By the convention used in std.t, this location is called
 *   "startroom".
 *   
 *   We'll make startroom an object of class "room", which is the basic
 *   class defined in adv.t for a location in the game.  
 */
startroom: room
    /* 
     *   The "sdesc" is the short description of the location.  This is
     *   displayed when the player enters or re-enters the location. 
     */
    sdesc = "Entryway"

    /* 
     *   The "ldesc" is the long description of the location.  This is
     *   displayed when the player enters the location for the first time,
     *   or types "look" while in the location. 
     */
    ldesc =
    {
        "This large, formal entryway is slightly intimidating:
        the walls are lined with somber portraits of gray-haired
        men from decades past; a medieval suit of armor";

        /* if the axe is in the armor, list it specially */
        if (axe.isIn(suitOfArmor))
            ", posed with battle axe at the ready,";

        " towers over a single straight-backed wooden chair.
        The front door leads back outside to the south.  A
        hallway leads north.";
    }

    /* 
     *   To the north is the hallway.  Set the "north" property to the
     *   destination room object.  Other direction properties that we
     *   could set: east, west, north, up, down, plus the diagonals: ne,
     *   nw, se, sw.  We can also set "in" and "out".  
     */
    north = hallway

    /* 
     *   To the south is the front door; set the "south" property to refer
     *   to the door object.
     */
    south = frontDoor

    /* 
     *   set the "out" direction to be the same as south, since going
     *   south leads outside 
     */
    out = frontDoor
;

/*
 *   Define the front door.
 */
frontDoor: doorway
    /* the door is in the starting room */
    location = startroom

    /* define the words the player can use to refer to the door */
    noun = 'door'
    adjective = 'front'

    /* the short description */
    sdesc = "front door"

    /*
     *   Define where the door goes if you go through it.  Normally, we'd
     *   just set this property to a room object.  In this case, though,
     *   we don't want the player to be able to go through the door, since
     *   this is one of the boundaries of the game.  So, when the player
     *   tries to go through the door, simply display a message about why
     *   they can't, then return "nil" - this is a special value in TADS
     *   that means that the destination is empty.  
     */
    doordest =
    {
        "You'd rather stay in the house for now. ";
        return nil;
    }
;

/*
 *   Define the chair.  The adv.t "chairitem" class is helpful here.
 */
chair: chairitem
    location = startroom
    sdesc = "wooden chair"
    noun = 'chair'
    adjective = 'straight-backed' 'wooden'
;

/*
 *   Define the suit of armor.  It can't be moved, so it's a "fixeditem"
 *   (defined in adv.t).  
 */
suitOfArmor: fixeditem
    location = startroom
    sdesc = "suit of armor"
    noun = 'suit' 'armor'
    adjective = 'medieval'
    ldesc =
    {
        "It's a suit of plate-mail armor that looks suitable
        for a very tall knight. ";

        /* if I'm holding the axe, so note */
        if (axe.isIn(self))
            "The armor is posed with a huge battle-axe held
            at the ready. ";
    }

    /* 
     *   This is a little unusual, but set the armor to be a "quiet
     *   container."  We initially have the axe located in the armor, even
     *   though the armor isn't a container in the normal sense.  The
     *   "qcontainer" flag will keep the axe from being listed separately
     *   in the contents of the room. 
     */
    isqcontainer = true
;

/*
 *   Battle Axe 
 */
axe: item
    sdesc = "battle axe"

    /* it starts out in the suit of armor */
    location = suitOfArmor

    /* 
     *   Define vocabulary.  Note that we provide some synonyms for things
     *   that really could be separate objects (the blade, the dried blood
     *   on the blade), but which aren't important enough to be
     *   implemented separately. 
     */
    noun = 'axe' 'blood' 'blade' 'edge'
    adjective = 'steel' 'battle' 'dried'
    ldesc = "It's a large steel battle axe.  A little bit of
             dried blood on the edge of the blade makes the authenticity
             of the equipment quite credible. "
;

/*
 *   Define the portraits.  We don't want to define several individual
 *   portraits, so define a single object that refers to the portraits
 *   collectively. 
 */
portraits: fixeditem
    location = startroom
    noun = 'portraits' 'pictures' 'men'
    adjective = 'somber' 'gray-haired'
    sdesc = "portraits"

    /* 
     *   set the "isThem" flag, so that default messages in adv.t refer to
     *   this object in the plural
     */
    isThem = true

    /* 
     *   Define an "adesc": this is used in messages to refer to the
     *   object with an indefinite pronoun (such as "a" or "an").
     *   Normally, the adv.t base class would define this as "a
     *   <<sdesc>>", but this isn't appropriate for a collective object -
     *   "a portraits" doesn't sound right.
     */
    adesc = "a portrait"
;

/*
 *   The hallway to the north of the starting room 
 */
hallway: room
    sdesc = "Hallway"
    ldesc = "This broad, dimly-lit corridor runs north and south. "

    south = startroom
    north = kitchen
;

kitchen: room
    sdesc = "Kitchen"
    ldesc = "This is a surprisingly cramped kitchen, equipped with
             antiques: the stove is a huge black iron contraption,
             and there doesn't even seem to be a refrigerator.  A
             hallway lies to the south. "
    south = hallway
;

/*
 *   The stove is fixed, since we don't want the player to be able to move
 *   it, and we'd also like it to be a container that can be opened and
 *   closed, so make it both a fixeditem and an openable.
 */
stove: fixeditem, openable
    location = kitchen
    noun = 'stove' 'oven' 'contraption' 'door'
    adjective = 'huge' 'black' 'iron' 'stove'
    sdesc = "stove"
    ldesc =
    {
        "It's a huge black iron cube, with a front door that swings
        open sideways.  The door is currently <<
        self.isopen ? "open" : "closed">>. ";

        /* list my contents if there's anything inside */
        if (self.isopen && itemcnt(self.contents) != 0)
            "Inside the stove you can see <<listcont(self)>>. ";
    }

    /* it starts off closed */
    isopen = nil
;

/*
 *   A fooditem is something that we can eat 
 */
loaf: fooditem
    location = stove
    sdesc = "loaf of bread"
    ldesc = "It's a fresh loaf with a golden-brown crust. "
    noun = 'loaf' 'bread' 'crust'
    adjective = 'fresh' 'golden-brown' 'brown'

    /* 
     *   override the "doEat" method - this is called when the player eats
     *   the item 
     */
    doEat(actor) =
    {
        "You tear off a piece and eat it; it's delicious.  You tear off
        a little more, then a little more, and before long the whole loaf
        is gone. ";

        /* make the bread vanish by moving it to "nil" */
        self.moveInto(nil);
    }
;

